<?php
/**
 * Plugin Name: Secure Cookie Handler
 * Description: Sets secure cookie attributes, disables XML-RPC, X-Frame-Options, X-Content-Type-Options, Referrer-Policy, Permissions-Policy, Content-Security-Policy, Strict-Transport-Security and optionally prevents browser refresh resubmission prompts.
 * Version: 1.1.0
 * Author: Cent Avenue
 * Author URI: https://centavenue.com/
 */

if (!defined('ABSPATH')) {
    exit;
}

class Secure_Cookie_Handler {
    private $option_name = 'secure_cookie_handler_options';

    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('init', [$this, 'validate_host_header'], 0);
        add_action('send_headers', [$this, 'send_security_headers']);

        add_action('init', function () {
    if (function_exists('header_remove')) {
        header_remove('X-Powered-By');
    }
}, 0);


        add_action('set_logged_in_cookie', [$this, 'apply_secure_cookie_settings'], 10, 5);
        add_action('login_enqueue_scripts', [$this, 'inject_form_resubmission_script']);

        add_filter('xmlrpc_enabled', [$this, 'maybe_disable_xmlrpc']);
    }

    public function add_admin_menu() {
        add_options_page(
            'Secure Cookie Settings',
            'Secure Cookies',
            'manage_options',
            'secure-cookie-handler',
            [$this, 'settings_page']
        );
    }

    public function register_settings() {
        register_setting($this->option_name, $this->option_name, [$this, 'sanitize_settings']);
    }

    public function sanitize_settings($input) {
        return [
            'domain'                  => sanitize_text_field($input['domain'] ?? ''),
            'path'                    => sanitize_text_field($input['path'] ?? '/'),
            'samesite'                => in_array($input['samesite'], ['Strict', 'Lax', 'None']) ? $input['samesite'] : 'Strict',
            'disable_xmlrpc'          => !empty($input['disable_xmlrpc']) ? 1 : 0,
            'form_resubmission_prompt'=> !empty($input['form_resubmission_prompt']) ? 1 : 0,
            'warning_sensitivity' => in_array($input['warning_sensitivity'] ?? '', ['low', 'strict', 'off']) ? $input['warning_sensitivity'] : 'low',

        // ✅ New header options
            'header_x_frame_options'     => !empty($input['header_x_frame_options']) ? 1 : 0,
            'header_x_content_type'      => !empty($input['header_x_content_type']) ? 1 : 0,
            'header_referrer_policy'     => !empty($input['header_referrer_policy']) ? 1 : 0,
            'header_permissions_policy'  => !empty($input['header_permissions_policy']) ? 1 : 0,
            'header_csp'                 => !empty($input['header_csp']) ? 1 : 0,
            'header_hsts'                => !empty($input['header_hsts']) ? 1 : 0,
        ];
    }

    public function settings_page() {
        $options = get_option($this->option_name);
        ?>
        
        <div class="wrap">
            <h1>Secure Cookie Handler Settings</h1>
            <!-- ✅ Header Status Check Panel -->
        <?php echo $this->get_cache_header_status(); ?>
            <form method="post" action="options.php">
                
                <?php settings_fields($this->option_name); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="domain">Cookie Domain</label></th>
                        <td><input type="text" name="<?= esc_attr($this->option_name); ?>[domain]" value="<?= esc_attr($options['domain'] ?? ''); ?>" class="regular-text" placeholder="e.g. localhost or example.com"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="path">Cookie Path</label></th>
                        <td><input type="text" name="<?= esc_attr($this->option_name); ?>[path]" value="<?= esc_attr($options['path'] ?? '/'); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="samesite">SameSite Attribute</label></th>
                        <td>
                            <select name="<?= esc_attr($this->option_name); ?>[samesite]">
                                <option value="Strict" <?= selected($options['samesite'] ?? '', 'Strict'); ?>>Strict</option>
                                <option value="Lax" <?= selected($options['samesite'] ?? '', 'Lax'); ?>>Lax</option>
                                <option value="None" <?= selected($options['samesite'] ?? '', 'None'); ?>>None</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                    <th scope="row"><label for="allowed_hosts">Allowed Hostnames</label></th>
                    <td>
                        <textarea name="<?= esc_attr($this->option_name); ?>[allowed_hosts]" rows="3" class="large-text code" placeholder="localhost, example.com"><?= esc_textarea($options['allowed_hosts'] ?? 'localhost'); ?></textarea>
                        <p class="description">Comma-separated list of valid hostnames. Requests with other Host headers will be blocked.</p>
                    </td>
                </tr>

                    <tr>
                        <th scope="row">Disable XML-RPC</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?= esc_attr($this->option_name); ?>[disable_xmlrpc]" value="1" <?= checked($options['disable_xmlrpc'] ?? 0, 1); ?>>
                                Completely disable XML-RPC functionality
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Enable Browser Refresh Prompt</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?= esc_attr($this->option_name); ?>[form_resubmission_prompt]" value="1" <?= checked($options['form_resubmission_prompt'] ?? 0, 1); ?>>
                                Warn user on login form refresh/resubmission
                            </label>
                        </td>
                    </tr>
                    <tr>
                    <th scope="row">Warning Sensitivity</th>
                    <td>
                        <select name="<?= esc_attr($this->option_name); ?>[warning_sensitivity]">
                            <option value="low" <?= selected($options['warning_sensitivity'] ?? '', 'low'); ?>>Low (Only on refresh/back)</option>
                            <option value="strict" <?= selected($options['warning_sensitivity'] ?? '', 'strict'); ?>>Strict (All unloads)</option>
                            <option value="off" <?= selected($options['warning_sensitivity'] ?? '', 'off'); ?>>Off (No warning)</option>
                        </select>
                        <p class="description">Controls how aggressively the browser warns users about leaving the login page with entered credentials.</p>
                    </td>
                </tr>
                <tr><th colspan="2"><h2>Security Headers</h2></th></tr>
                        <tr>
                            <th scope="row">X-Frame-Options</th>
                            <td><label><input type="checkbox" name="<?= esc_attr($this->option_name); ?>[header_x_frame_options]" value="1" <?= checked($options['header_x_frame_options'] ?? 0, 1); ?>> Enable (DENY)</label></td>
                        </tr>
                        <tr>
                            <th scope="row">X-Content-Type-Options</th>
                            <td><label><input type="checkbox" name="<?= esc_attr($this->option_name); ?>[header_x_content_type]" value="1" <?= checked($options['header_x_content_type'] ?? 0, 1); ?>> Enable (nosniff)</label></td>
                        </tr>
                        <tr>
                            <th scope="row">Referrer-Policy</th>
                            <td><label><input type="checkbox" name="<?= esc_attr($this->option_name); ?>[header_referrer_policy]" value="1" <?= checked($options['header_referrer_policy'] ?? 0, 1); ?>> Enable (strict-origin-when-cross-origin)</label></td>
                        </tr>
                        <tr>
                            <th scope="row">Permissions-Policy</th>
                            <td><label><input type="checkbox" name="<?= esc_attr($this->option_name); ?>[header_permissions_policy]" value="1" <?= checked($options['header_permissions_policy'] ?? 0, 1); ?>> Enable (basic)</label></td>
                        </tr>
                        <tr>
                            <th scope="row">Content-Security-Policy</th>
                            <td><label><input type="checkbox" name="<?= esc_attr($this->option_name); ?>[header_csp]" value="1" <?= checked($options['header_csp'] ?? 0, 1); ?>> Enable (basic self policy)</label></td>
                        </tr>
                        <tr>
                            <th scope="row">Strict-Transport-Security (HTTPS only)</th>
                            <td><label><input type="checkbox" name="<?= esc_attr($this->option_name); ?>[header_hsts]" value="1" <?= checked($options['header_hsts'] ?? 0, 1); ?>> Enable (max-age=31536000; includeSubDomains)</label></td>
                        </tr>


                </table>
                <?php submit_button(); ?>
            </form>
            
        </div>
        <?php
    }

    public function apply_secure_cookie_settings($cookie, $expire, $expiration, $user_id, $scheme) {
        $options = get_option($this->option_name);
        $domain = $options['domain'] ?? $_SERVER['HTTP_HOST'];
        $path = $options['path'] ?? '/';
        $samesite = in_array($options['samesite'] ?? '', ['Strict', 'Lax', 'None']) ? $options['samesite'] : 'Strict';

        list($name, $value) = explode('=', $cookie, 2);

        setcookie($name, $value, [
            'expires'  => $expire,
            'path'     => $path,
            'domain'   => $domain,
            'secure'   => is_ssl(),
            'httponly' => true,
            'samesite' => $samesite
        ]);
    }

    public function inject_form_resubmission_script() {
    $opts = get_option($this->option_name);

    if (!empty($opts['form_resubmission_prompt']) && ($opts['warning_sensitivity'] ?? 'low') !== 'off') {
        $sensitivity = $opts['warning_sensitivity'] ?? 'low';
        ?>
        <script>
        document.addEventListener('DOMContentLoaded', function () {
            let loginForm = document.getElementById('loginform');
            let isSubmitting = false;

            if (loginForm) {
                loginForm.addEventListener('submit', function () {
                    isSubmitting = true;
                    // clear POST state after submit to avoid resubmission dialog
                    if (window.history.replaceState) {
                        window.history.replaceState(null, null, window.location.href);
                    }
                });

                if ("<?= esc_js($sensitivity); ?>" === 'strict') {
                    window.addEventListener('beforeunload', function (e) {
                        if (!isSubmitting) {
                            e.preventDefault();
                            e.returnValue = "⚠ You may be leaving with unsaved login form data.";
                        }
                    });
                } else if ("<?= esc_js($sensitivity); ?>" === 'low') {
                    window.addEventListener('beforeunload', function (e) {
                        if (!isSubmitting && performance.navigation.type === 1) {
                            e.preventDefault();
                            e.returnValue = "⚠ You may be resubmitting your login form.";
                        }
                    });
                }
            }
        });
        </script>
        <?php
    }
}
public function validate_host_header() {
    if ( is_admin() || defined('DOING_AJAX') || php_sapi_name() === 'cli' ) {
        return;
    }

    $options = get_option($this->option_name);
    $allowed = isset($options['allowed_hosts']) ? array_map('trim', explode(',', $options['allowed_hosts'])) : [];

    $host = $_SERVER['HTTP_HOST'] ?? '';
    $host = strtolower(trim($host));

    if (!empty($allowed) && !in_array($host, $allowed, true)) {
        header('HTTP/1.1 400 Bad Request');
        header('Content-Type: text/plain; charset=utf-8');
        echo "Invalid Host header: " . htmlentities($host);
        exit;
    }
}

public function get_cache_header_status(): string {
    $headers = headers_list();
    $required = [
        'Cache-Control' => ['no-store', 'no-cache', 'must-revalidate'],
        'Pragma'        => ['no-cache'],
        'Expires'       => ['0', 'Wed, 11 Jan 1984 05:00:00 GMT']
    ];

    $status = [];

    foreach ($required as $header => $required_values) {
        $found = false;
        foreach ($headers as $sent_header) {
            if (stripos($sent_header, $header . ':') === 0) {
                foreach ($required_values as $val) {
                    if (stripos($sent_header, $val) !== false) {
                        $found = true;
                        break;
                    }
                }
            }
        }
        $status[$header] = $found;
    }

    $all_ok = !in_array(false, $status, true);

    ob_start();
    ?>
    <div style="padding: 10px; margin: 15px 0; border-left: 5px solid <?= $all_ok ? '#28a745' : '#dc3545'; ?>; background: #f9f9f9;">
        <strong>Browser Cache Header Status:</strong><br>
        <?php foreach ($status as $header => $ok): ?>
            <p style="margin: 4px 0;">
                <?= esc_html($header); ?>:
                <span style="color: <?= $ok ? 'green' : 'red'; ?>;">
                    <?= $ok ? '✅ Present' : '❌ Missing'; ?>
                </span>
            </p>
        <?php endforeach; ?>
    </div>
    <?php
    return ob_get_clean();
}
public function send_security_headers() {
    $opts = get_option($this->option_name);

    if (!headers_sent()) {
        if (!empty($opts['header_x_frame_options'])) {
            header('X-Frame-Options: DENY');
        }

        if (!empty($opts['header_x_content_type'])) {
            header('X-Content-Type-Options: nosniff');
        }

        if (!empty($opts['header_referrer_policy'])) {
            header('Referrer-Policy: strict-origin-when-cross-origin');
        }

        if (!empty($opts['header_permissions_policy'])) {
            header('Permissions-Policy: geolocation=(), microphone=(), camera=()');
        }

        if (!empty($opts['header_csp'])) {
            header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline';");
        }

        if (!empty($opts['header_hsts']) && is_ssl()) {
            header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
        }
    }
}


    public function maybe_disable_xmlrpc() {
        $options = get_option($this->option_name);
        return empty($options['disable_xmlrpc']) ? true : false;
    }
}

new Secure_Cookie_Handler();
