<?php
/**
 * Prevent XSS Vulnerability Reflected Settings.
 *
 * @package PreventXSSVulnerability
 */

/**
 * Create HTML of Reflected Settings page.
 */
class Prevent_XSS_Vulnerability_Reflected_Settings {
	/**
	 * Call Reflected Settings Function.
	 */
	public function __construct() {
		$this->xss_reflected_settings();
	}

	/**
	 * Generate Block entities section HTML.
	 *
	 * @access private
	 * @since 1.0.0
	 *
	 * @param string $excluded_entities Exclude entities from being block.
	 * @param string $include_entities  Include entities to be blocked.
	 * @param string $enabled           Whether the block is enabled or not.
	 *
	 * @return void
	 */
	private function get_block_output( $excluded_entities, $include_entities, $enabled ) {
		?>
		<table class="prevent-xss reflected-xss">
			<caption>
				<?php
				esc_html_e(
					'Block Entities (Remove Entities)',
					'prevent-xss-vulnerability'
				);
				?>
			</caption>
			<tbody>
			<tr>
				<td>
				<p>
					<?php
					esc_html_e(
						'This plugin blocks or removes the following entities from website URLs:',
						'prevent-xss-vulnerability'
					);
					?>
				</p>
				<ol class="reflected-xss-items">
					<li>
						<?php esc_html_e( 'Opening Round Bracket', 'prevent-xss-vulnerability' ); ?>
						<code>(</code>
					</li>
					<li>
						<?php esc_html_e( 'Closing Round Bracket', 'prevent-xss-vulnerability' ); ?>  <code>)</code>
					</li>
					<li>
						<?php esc_html_e( 'Less than Sign', 'prevent-xss-vulnerability' ); ?> <code>&lt;</code>
					</li>
					<li>
						<?php esc_html_e( 'Greater than Sign', 'prevent-xss-vulnerability' ); ?> <code>&gt;</code>
					</li>
					<li>
						<?php esc_html_e( 'Opening Square Bracket', 'prevent-xss-vulnerability' ); ?> <code>[</code>
					</li>
					<li>
						<?php esc_html_e( 'Closing Square Bracket', 'prevent-xss-vulnerability' ); ?> <code>]</code>
					</li>
					<li>
						<?php esc_html_e( 'Opening Curly Bracket', 'prevent-xss-vulnerability' ); ?> <code>{</code>
					</li>
					<li>
						<?php esc_html_e( 'Pipe or Vertical Bar', 'prevent-xss-vulnerability' ); ?> <code>|</code>
					</li>
					<li>
						<?php esc_html_e( 'Closing Curly Bracket', 'prevent-xss-vulnerability' ); ?> <code>}</code>
					</li>
				</ol>
				</td>
			</tr>
			<tr>
				<td>
				<b>
					<label for="include_extra_entities">
						<?php esc_html_e( 'Include Extra Entities to be blocked:', 'prevent-xss-vulnerability' ); ?>
					</label>
				</b>
				<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
				<textarea name="include_extra_entities" placeholder="<?php esc_html_e( 'Add Characcters/Entities in Comma Separated Form', 'prevent-xss-vulnerability' ); ?>" rows="5" cols="100"><?php esc_html_e( $include_entities ); ?></textarea>
				<small>
					<?php esc_html_e( 'Add Characcters/Entities in Comma Separated that you want to be blocked/remove in the URL. Like:', 'prevent-xss-vulnerability' ); ?> <code>~</code>,<code>$</code>,<code>%</code>
				</small>
				</td>
			</tr>
			<tr>
				<td>
				<b>
					<label for="exclude_block_entities">
						<?php esc_html_e( 'Exclude pre-defined Entities from being blocked:', 'prevent-xss-vulnerability' ); ?>
					</label>
				</b>
				<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
				<textarea name="exclude_block_entities" placeholder="<?php esc_html_e( 'Add Entities in Comma Separated Form', 'prevent-xss-vulnerability' ); ?>" rows="5" cols="100"><?php esc_html_e( $excluded_entities ); ?></textarea>
				<small>
					<?php esc_html_e( 'Add Entities in Comma Separated that you do not want to be blocked/remove in the URL. Like:', 'prevent-xss-vulnerability' ); ?> <code>[</code>,<code>{</code>,<code>]</code>
				</small>
				</td>
			</tr>
			<tr>
				<td class="enable">
					<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
					<input type="checkbox" name="enable_blocking" value=1 <?php esc_html_e( $enabled ); ?>/>
					<b>
						<?php esc_html_e( 'Enable Blocking', 'prevent-xss-vulnerability' ); ?>
					</b>
				</td>
			</tr>
			</tbody>
		</table>

		<?php
	}

	/**
	 * Generate Encode entities section HTML.
	 *
	 * @access private
	 * @since 1.0.0
	 *
	 * @param string $excluded_entities Excluded entities from being encode.
	 * @param string $enabled           Whether the encode is enabled or not.
	 *
	 * @return void
	 */
	private function get_encode_output( $excluded_entities, $enabled ) {
		?>
		<table class="prevent-xss reflected-xss">
			<caption>
			<?php esc_html_e( 'Encode Entities', 'prevent-xss-vulnerability' ); ?>
			</caption>
			<tbody>
			<tr>
				<td>
				<p>
					<?php
						esc_html_e(
							'This plugin encodes the following entities within website URLs:',
							'prevent-xss-vulnerability'
						);
					?>
				</p>
				<ol class="reflected-xss-items">
					<li>
						<?php
							esc_html_e(
								'Exclamation Mark',
								'prevent-xss-vulnerability'
							);
						?>
						<code>!</code>
					</li>
					<li>
						<?php
							esc_html_e(
								'Double Quotation',
								'prevent-xss-vulnerability'
							);
						?>
						<code>"</code>
					</li>
					<li>
						<?php
							esc_html_e(
								'Single Quotation',
								'prevent-xss-vulnerability'
							);
						?>
						<code>\</code>
					</li>
					<li>
						<?php
							esc_html_e(
								'Opening Round Bracket',
								'prevent-xss-vulnerability'
							);
						?>
						<code>(</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Closing Round Bracket', 'prevent-xss-vulnerability' );
						?>
						<code>)</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Asterisk Sign', 'prevent-xss-vulnerability' );
						?>
						<code>*</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Less than Sign', 'prevent-xss-vulnerability' );
						?>
						<code>&lt;</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Greater than Sign', 'prevent-xss-vulnerability' );
						?>
						<code>&gt;</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Grave Accent', 'prevent-xss-vulnerability' );
						?>
						<code>`</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Cap Sign', 'prevent-xss-vulnerability' );
						?>
						<code>^</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Opening Square Bracket', 'prevent-xss-vulnerability' );
						?>
						<code>[</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Closing Square Bracket', 'prevent-xss-vulnerability' );
						?>
						<code>]</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Opening Curly Bracket', 'prevent-xss-vulnerability' );
						?>
						<code>{</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Pipe or Vertical Bar', 'prevent-xss-vulnerability' );
						?>
						<code>|</code>
					</li>
					<li>
						<?php
							esc_html_e( 'Closing Curly Bracket', 'prevent-xss-vulnerability' );
						?>
						<code>}</code>
					</li>
				</ol>
				</td>
			</tr>
			<tr>
				<td>
				<b>
					<label for="exclude_encoding_entities">
						<?php
							esc_html_e(
								'Exclude Entities from Encoding: ',
								'prevent-xss-vulnerability'
							);
						?>
					</label>
				</b>
				<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
				<textarea name="exclude_encode_entities" placeholder="<?php esc_html_e( 'Add Entities in Comma Separated Form', 'prevent-xss-vulnerability' ); ?>" rows="5" cols="100"><?php esc_html_e( $excluded_entities ); ?></textarea>
				<small>
					<?php
						esc_html_e(
							'Add Entities in Comma Separated list that you do not want to be encoded in the URL. Like:',
							'prevent-xss-vulnerability'
						);
					?>
					<code>[</code>,<code>]</code>
				</small>
				</td>
			</tr>
			<tr>
				<td class="enable">
				<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
				<input type="checkbox" name="enable_encoding" value=1 <?php esc_html_e( $enabled ); ?>/>
				<b> <?php esc_html_e( 'Enable Encoding', 'prevent-xss-vulnerability' ); ?> </b>
				</td>
			</tr>
			</tbody>
		</table>

		<?php
	}

	/**
	 * Generate Escape section HTML.
	 *
	 * @access private
	 * @since 1.0.0
	 *
	 * @param string $enabled Whether the escape is enabled or not.
	 */
	private function get_escape_output( $enabled ) {
		?>
		<table class="prevent-xss reflected-xss">
			<caption>
				<?php esc_html_e( 'Escape HTML', 'prevent-xss-vulnerability' ); ?>
			</caption>
			<tbody>
			<tr>
				<td>
					<p>
						<?php
							esc_html_e(
								'This plugin strips HTML tags from the $_GET variable in PHP. Consequently, any part of your theme or plugin that utilizes $_GET to retrieve data from the URL will receive the HTML-escaped value. This effectively prevents XSS attacks that exploit HTML and JavaScript events.',
								'prevent-xss-vulnerability'
							);
						?>
					</p>

					<p>
						<?php
							esc_html_e(
								'Important Note: If any plugin or custom code directly fetches URL parameters without using the $_GET variable, the retrieved values will not be HTML-escaped by this plugin.',
								'prevent-xss-vulnerability'
							);
						?>
					</p>
				</td>
			</tr>
			<tr>
				<td class="enable">
					<?php // phpcs:ignore WordPress.WP.I18n.NonSingularStringLiteralText ?>
					<input type="checkbox" name="escape_html" value=1 <?php esc_html_e( $enabled ); ?>/>
					<b>
					<?php esc_html_e( 'Enable Escaping', 'prevent-xss-vulnerability' ); ?>
					</b>
				</td>
			</tr>
			</tbody>
		</table>

		<?php
	}

	/**
	 * Generate Reflected page note HTML.
	 *
	 * @access private
	 * @since 1.0.0
	 */
	private function get_reflected_desc() {
		?>
		<div>
			<p>
				<?php
					esc_html_e(
						'Reflected XSS occurs when malicious code is injected into a website\'s URL. This code can then be executed by a user\'s browser, potentially stealing information or compromising their system. For more information on XSS, please visit',
						'prevent-xss-vulnerability'
					);
				?>
				<a href="https://www.yasglobal.com/cross-site-scripting-xss/" title="Cross-site Scripting (XSS)" target="_blank"><?php esc_html_e( 'here', 'prevent-xss-vulnerability' ); ?></a>.
			</p>
		</div>

		<table class="important">
			<tr>
			<td class="icon">
				<i class="fa icon icon-important" title="important">&nbsp;</i>
			</td>
			<td>
				<p>
					<?php
					esc_html_e(
						'After configuring these settings, please thoroughly test your website by randomly navigating through different pages to ensure all functionalities are working as expected.',
						'prevent-xss-vulnerability'
					);
					?>
				</p>
				<p>
					<?php
					esc_html_e(
						'If you are using WooCommerce, it is crucial to test the entire product purchase process, including cart, checkout, and order completion, to verify that the plugin does not interfere with any core functionalities of your online store.',
						'prevent-xss-vulnerability'
					);
					?>
				</p>
			</td>
			</tr>
		</table>

		<?php
	}

	/**
	 * Save Reflected Settings.
	 *
	 * @access private
	 * @since 1.0.0
	 */
	private function save_reflected_settings() {
		$form_submit = filter_input( INPUT_POST, 'submit' );
		$user_id     = get_current_user_id();

		if ( $form_submit
			&& check_admin_referer(
				'prevent-xss-vulnerability-reflected_' . $user_id,
				'_prevent_xss_vulnerability_reflected_nonce'
			)
		) {
			$reflected_xss = array(
				'exclude_encode_entities' => '',
				'enable_encoding'         => 0,
				'exclude_block_entities'  => '',
				'include_extra_entities'  => '',
				'enable_blocking'         => 0,
				'escape_html'             => 0,
			);

			$encode_entities = filter_input( INPUT_POST, 'exclude_encode_entities' );
			if ( $encode_entities ) {
				$reflected_xss['exclude_encode_entities'] = wp_kses(
					$encode_entities,
					array()
				);
			}

			$enable_encoding = (int) filter_input( INPUT_POST, 'enable_encoding' );
			if ( $enable_encoding && 1 === $enable_encoding ) {
				$reflected_xss['enable_encoding'] = 1;
			}

			$block_entities = filter_input( INPUT_POST, 'exclude_block_entities' );
			if ( $block_entities ) {
				$reflected_xss['exclude_block_entities'] = wp_kses(
					$block_entities,
					array()
				);
			}

			$include_entities = filter_input( INPUT_POST, 'include_extra_entities' );
			if ( $include_entities ) {
				$reflected_xss['include_extra_entities'] = wp_kses(
					$include_entities,
					array()
				);
			}

			$enable_blocking = (int) filter_input( INPUT_POST, 'enable_blocking' );
			if ( $enable_blocking && 1 === $enable_blocking ) {
				$reflected_xss['enable_blocking'] = 1;
			}

			$escape_html = (int) filter_input( INPUT_POST, 'escape_html' );
			if ( $escape_html && 1 === $escape_html ) {
				$reflected_xss['escape_html'] = 1;
			}

			update_option(
				'prevent_xss_vulnerability_reflected_settings',
				$reflected_xss
			);
		}
	}

	/**
	 * Reflected Settings Page.
	 *
	 * @access private
	 * @since 0.1
	 */
	private function xss_reflected_settings() {
		$this->save_reflected_settings();

		$user_id      = get_current_user_id();
		$xss_settings = get_option( 'prevent_xss_vulnerability_reflected_settings' );

		if ( is_string( $xss_settings ) ) {
			$xss_settings = maybe_unserialize( $xss_settings );
		}

		$enable_blocking     = '';
		$enable_encoding     = '';
		$escape_html         = '';
		$xss_extra_entities  = '';
		$xss_block_entities  = '';
		$xss_encode_entities = '';

		if ( isset( $xss_settings['exclude_block_entities'] ) ) {
			$xss_block_entities = $xss_settings['exclude_block_entities'];
		}

		if ( isset( $xss_settings['include_extra_entities'] ) ) {
			$xss_extra_entities = $xss_settings['include_extra_entities'];
		}

		if ( isset( $xss_settings['enable_blocking'] )
			&& 1 === $xss_settings['enable_blocking']
		) {
			$enable_blocking = 'checked';
		}

		if ( isset( $xss_settings['exclude_encode_entities'] ) ) {
			$xss_encode_entities = $xss_settings['exclude_encode_entities'];
		}

		if ( isset( $xss_settings['enable_encoding'] )
			&& 1 === $xss_settings['enable_encoding']
		) {
			$enable_encoding = 'checked';
		}

		if ( isset( $xss_settings['escape_html'] )
			&& 1 === $xss_settings['escape_html']
		) {
			$escape_html = 'checked';
		}
		?>

		<div class="wrap">
			<h2>
			<?php
			esc_html_e( 'Reflected Cross-site Scripting (XSS) Settings', 'prevent-xss-vulnerability' );
			?>
			</h2>

			<?php $this->get_reflected_desc(); ?>

			<form enctype="multipart/form-data" action="" method="POST" id="reflected-xss">
			<?php
				wp_nonce_field(
					'prevent-xss-vulnerability-reflected_' . $user_id,
					'_prevent_xss_vulnerability_reflected_nonce',
					true
				);
				$this->get_block_output(
					$xss_block_entities,
					$xss_extra_entities,
					$enable_blocking
				);
				$this->get_encode_output( $xss_encode_entities, $enable_encoding );
				$this->get_escape_output( $escape_html );
			?>
			<p class="submit">
				<input type="submit" name="submit" id="submit" class="button button-primary" value="<?php esc_html_e( 'Save Changes', 'prevent-xss-vulnerability' ); ?>" />
			</p>
			</form>
		</div>
		<?php
	}
}
