<?php
/**
 * Prevent XSS Vulnerability Admin.
 *
 * @package PreventXSSVulnerability
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Create Admin menu for plugin settings, attach styles etc.
 */
class Prevent_XSS_Vulnerability_Admin {
	/**
	 * Css file suffix (version number with with extension).
	 *
	 * @var string
	 */
	private $css_suffix = '-' . PREVENT_XSS_VULNERABILITY_VERSION . '.min.css';

	/**
	 * Initializes WordPress hooks.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
	}

	/**
	 * Added Pages in Menu for Settings.
	 *
	 * @access public
	 * @since 0.1
	 *
	 * @return void
	 */
	public function admin_menu() {
		add_menu_page(
			'Prevent XSS Vulnerability',
			'Prevent XSS Vulnerability',
			'activate_plugins',
			'prevent-xss-vulnerability-reflected-settings',
			array( $this, 'reflected_settings' ),
			'dashicons-shield'
		);
		$reflected_page = add_submenu_page(
			'prevent-xss-vulnerability-reflected-settings',
			'Reflected Cross-site scripting Settings',
			'Reflected XSS',
			'activate_plugins',
			'prevent-xss-vulnerability-reflected-settings',
			array( $this, 'reflected_settings' )
		);
		$self_page      = add_submenu_page(
			'prevent-xss-vulnerability-reflected-settings',
			'Self Cross-site scripting Settings',
			'Self-XSS',
			'activate_plugins',
			'prevent-xss-vulnerability-self-settings',
			array( $this, 'self_xss_settings' )
		);
		$about_page     = add_submenu_page(
			'prevent-xss-vulnerability-reflected-settings',
			'About Cross-site scripting',
			'About',
			'activate_plugins',
			'prevent-xss-vulnerability-about',
			array( $this, 'about_xss' )
		);

		add_action(
			'admin_print_styles-' . $reflected_page . '',
			array( $this, 'add_settings_page_style' )
		);
		add_action(
			'admin_print_styles-' . $self_page . '',
			array( $this, 'add_settings_page_style' )
		);
		add_action(
			'admin_print_styles-' . $about_page . '',
			array( $this, 'add_about_style' )
		);

		add_filter(
			'plugin_action_links_' . PREVENT_XSS_VULNERABILITY_BASENAME,
			array( $this, 'settings_link' )
		);
	}

	/**
	 * Add about page style.
	 */
	public function add_about_style() {
		wp_enqueue_style(
			'prevent-xss-vulnerability-about-style',
			plugins_url(
				'/assets/css/about-plugins' . $this->css_suffix,
				PREVENT_XSS_VULNERABILITY_FILE
			),
			array(),
			true
		);
	}

	/**
	 * Add settings page style.
	 */
	public function add_settings_page_style() {
		wp_enqueue_style(
			'prevent-xss-vulnerability-settings-style',
			plugins_url(
				'/assets/css/admin-style' . $this->css_suffix,
				PREVENT_XSS_VULNERABILITY_FILE
			),
			array(),
			true
		);
	}

	/**
	 * Settings Page where user can change the Settings for XSS (Choose which params
	 * to encode, which to block, Escape HTML).
	 *
	 * @access public
	 * @since 0.1
	 *
	 * @return void
	 */
	public function reflected_settings() {
		include_once PREVENT_XSS_VULNERABILITY_PATH .
			'admin/class-prevent-xss-vulnerability-reflected-settings.php';
		new Prevent_XSS_Vulnerability_Reflected_Settings();

		add_filter( 'admin_footer_text', array( $this, 'admin_footer_text' ), 1 );
	}

	/**
	 * Settings Page where user can change the Settings for XSS (Choose which params
	 * to encode, which to block, Escape HTML).
	 *
	 * @access public
	 * @since 0.3.0
	 *
	 * @return void
	 */
	public function self_xss_settings() {
		include_once PREVENT_XSS_VULNERABILITY_PATH .
			'admin/class-prevent-xss-vulnerability-self-settings.php';
		new Prevent_XSS_Vulnerability_Self_Settings();

		add_filter( 'admin_footer_text', array( $this, 'admin_footer_text' ), 1 );
	}

	/**
	 * Add About Plugins Page.
	 *
	 * @access public
	 * @since 0.1
	 *
	 * @return void
	 */
	public function about_xss() {
		include_once PREVENT_XSS_VULNERABILITY_PATH .
			'admin/class-prevent-xss-vulnerability-about.php';
		new Prevent_XSS_Vulnerability_About();

		add_filter( 'admin_footer_text', array( $this, 'admin_footer_text' ), 1 );
	}

	/**
	 * Add Plugin Support and Follow Message in the footer of Admin Pages.
	 *
	 * @access public
	 * @since 0.2
	 *
	 * @return string
	 */
	public function admin_footer_text() {
		$footer_text = sprintf(
			// translators: placeholders like %2$s replaced with the link.
			__( 'Prevent XSS Vulnerability version %1$s by <a href="%2$s" title="YAS Global Website" target="_blank">Sami Ahmed Siddiqui</a> - <a href="%3$s" title="Support forums" target="_blank">Support forums</a> - Follow on Twitter: <a href="%4$s" title="Follow Sami Ahmed Siddiqui on Twitter" target="_blank">Sami Ahmed Siddiqui</a>', 'prevent-xss-vulnerability' ),
			PREVENT_XSS_VULNERABILITY_VERSION,
			'https://www.yasglobal.com',
			'https://wordpress.org/support/plugin/prevent-xss-vulnerability',
			'https://twitter.com/samisiddiqui91'
		);

		return $footer_text;
	}

	/**
	 * Plugin About, Contact and Settings Link on the Plugin Page under
	 * the Plugin Name.
	 *
	 * @access public
	 * @since 0.1
	 *
	 * @param  array $links Contains the Plugin Basic Link (Activate/Deactivate/Delete).
	 *
	 * @return array Returns the Plugin Basic Links and added some custome link
	 *               for Settings, Contact and About.
	 */
	public function settings_link( $links ) {
		$about = sprintf(
			// translators: %s replaced with the link.
			__( '<a href="%s" title="About">About</a>', 'prevent-xss-vulnerability' ),
			'admin.php?page=prevent-xss-vulnerability-about'
		);
		$contact = sprintf(
			// translators: %s replaced with the link.
			__( '<a href="%s" title="Contact">Contact</a>', 'prevent-xss-vulnerability' ),
			'https://www.yasglobal.com/#request-form'
		);
		$settings_link = sprintf(
			// translators: %s replaced with the link.
			__( '<a href="%s" title="Settings">Settings</a>', 'prevent-xss-vulnerability' ),
			'admin.php?page=prevent-xss-vulnerability-reflected-settings'
		);

		array_unshift( $links, $settings_link );
		array_unshift( $links, $contact );
		array_unshift( $links, $about );

		return $links;
	}
}

new Prevent_XSS_Vulnerability_Admin();
