<?php
/*
Plugin Name: Malicious File Upload Prevention
Plugin URI: https://centavenue.com/
Description: A plugin to prevent malicious file uploads by validating file size, MIME type, file extension, and file content.
Version: 1.1
Author: Cent Avenue
Author URI: https://centavenue.com/
License: GPL2
*/

// Hook into the file upload prefilter
add_filter('wp_handle_upload_prefilter', 'mfu_validate_file_upload');

function mfu_validate_file_upload($file) {
    // Validate file size (limit to 15MB)
    $max_file_size = 15 * 1024 * 1024; // 2MB in bytes
    if ($file['size'] > $max_file_size) {
        $file['error'] = 'File size exceeds the 15MB limit.';
        return $file;
    }

    // Validate file extension
    $allowed_extensions = array('jpg', 'jpeg', 'png', 'gif', 'pdf');
    $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    if (!in_array(strtolower($file_extension), $allowed_extensions)) {
        $file['error'] = 'Invalid file extension. Only JPG, PNG, GIF, and PDF files are allowed.';
        return $file;
    }

    // Validate MIME type
    $allowed_mimes = array(
        'image/jpeg',
        'image/png',
        'image/gif',
        'application/pdf'
    );
    if (!in_array($file['type'], $allowed_mimes)) {
        $file['error'] = 'Invalid file MIME type.';
        return $file;
    }

    // Validate file content (ensure it’s an actual image or PDF)
    if (in_array($file_extension, array('jpg', 'jpeg', 'png', 'gif'))) {
        $image_info = @getimagesize($file['tmp_name']);
        if ($image_info === false) {
            $file['error'] = 'Uploaded file is not a valid image.';
            return $file;
        }
    } elseif ($file_extension === 'pdf') {
    $file_type = mime_content_type($file['tmp_name']);
    if ($file_type !== 'application/pdf') {
        $file['error'] = 'Uploaded file is not a valid PDF.';
        return $file;
    }

    // Read a reasonable chunk of the file to detect embedded JavaScript
    $pdf_content = file_get_contents($file['tmp_name'], false, null, 0, 8192); // 8KB

    // Detect dangerous embedded PDF JavaScript or actions
    if (preg_match('/\/(JavaScript|JS|AA|OpenAction)/i', $pdf_content) ||
        preg_match('/app\.alert\s*\(|this\.exportDataObject/i', $pdf_content)) {
        $file['error'] = 'The PDF contains embedded JavaScript or suspicious actions, which are not allowed.';
        return $file;
    }
}



    return $file;
}

// Block executable file uploads by disabling certain MIME types
add_filter('upload_mimes', 'mfu_block_executable_files');

function mfu_block_executable_files($mimes) {
    // Remove dangerous file types
    unset($mimes['php']);
    unset($mimes['exe']);
    unset($mimes['js']);
    unset($mimes['sh']);
    unset($mimes['bat']);
    
    return $mimes;
}

// Prevent PHP execution in the uploads folder by creating a .htaccess file
register_activation_hook(__FILE__, 'mfu_add_htaccess_to_uploads');

function mfu_add_htaccess_to_uploads() {
    $uploads_dir = wp_upload_dir();
    $htaccess_path = $uploads_dir['basedir'] . '/.htaccess';

    if (!file_exists($htaccess_path)) {
        $htaccess_content = " 
# Disable PHP execution
<Files *.php>
    deny from all
</Files>";
        file_put_contents($htaccess_path, $htaccess_content);
    }
}
