<?php
/*
Plugin Name: Dispensary Tables Viewer
Description: Display custom MySQL tables with pagination, search, and entries dropdown using shortcodes.
Version: 3.0
*/

if (!defined('ABSPATH')) exit;

// ======================================================
// GENERIC TABLE VIEW FUNCTION (FIXED PAGINATION)
// ======================================================
function disp_generic_table_viewer($table_name, $columns, $default_per_page = 10) {
    global $wpdb;

    // Remove "category" column if exists
    $columns = array_filter($columns, function($c){
        return strtolower($c) !== "category";
    });

    // Records per page
    $per_page = isset($_GET['entries']) ? max(intval($_GET['entries']), 1) : $default_per_page;

    // Detect current page correctly (shortcode compatible)
    $page = get_query_var('paged') ? get_query_var('paged') :
            (get_query_var('page') ? get_query_var('page') :
            (isset($_GET['paged']) ? intval($_GET['paged']) : 1));

    if ($page < 1) $page = 1;

    // Search keyword
    $search = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';

    // WHERE clause
    $where = "";
    $params = [];

    if ($search !== '') {
        $like = '%' . $wpdb->esc_like($search) . '%';
        $conditions = [];

        foreach ($columns as $c) {
            $conditions[] = "$c LIKE %s";
            $params[] = $like;
        }

        $where = "WHERE " . implode(" OR ", $conditions);
    }

    // Total rows count
    if ($search !== '') {
        $total = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name $where", $params
        ));
    } else {
        $total = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
    }

    // Pagination offset
    $offset = ($page - 1) * $per_page;

    // Fetch rows
    if ($search !== '') {
        $query = $wpdb->prepare(
            "SELECT * FROM $table_name $where LIMIT %d, %d",
            array_merge($params, [$offset, $per_page])
        );
    } else {
        $query = $wpdb->prepare(
            "SELECT * FROM $table_name LIMIT %d, %d",
            [$offset, $per_page]
        );
    }

    $rows = $wpdb->get_results($query);

    // =====================================================
    // OUTPUT HTML
    // =====================================================
    ob_start();

    ?>
    <style>
        .disp-wrapper { margin-top:20px; }
        .disp-top-bar { display:flex; justify-content:space-between; margin-bottom:10px; }
        .disp-search input { padding:6px; width:180px; }
        .disp-entries select { padding:6px; }
        table.disp-table { width:100%; border-collapse:collapse; }
        table.disp-table th, table.disp-table td { border:1px solid #ccc; padding:8px; }
        table.disp-table th { background:#f5f5f5; }
        .disp-pagination { margin-top:15px; text-align:right; }
        .disp-pagination a {
            padding:6px 10px;
            margin:0 3px;
            border:1px solid #333;
            text-decoration:none;
        }
        .disp-pagination .current {
            background:#333;
            color:#fff;
            font-weight:bold;
        }
    </style>

    <div class="disp-wrapper">

        <form method="GET" class="disp-top-bar">

            <!-- LEFT: Entries per page -->
            <div class="disp-entries">
                Show
                <select name="entries" onchange="this.form.submit()">
                    <?php foreach ([10,25,50] as $e): ?>
                        <option value="<?php echo $e; ?>" <?php selected($per_page, $e); ?>>
                            <?php echo $e; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                entries
            </div>

            <!-- RIGHT: Search box -->
            <div class="disp-search">
                <input type="text" name="search" placeholder="Search..." value="<?php echo esc_attr($search); ?>">
            </div>

        </form>

        <!-- TABLE -->
        <table class="disp-table">
            <thead>
                <tr>
                    <?php foreach ($columns as $col): ?>
                        <th><?php echo esc_html(ucwords(str_replace("_", " ", $col))); ?></th>
                    <?php endforeach; ?>
                </tr>
            </thead>
            <tbody>
                <?php if ($rows): ?>
                    <?php foreach ($rows as $r): ?>
                        <tr>
                            <?php foreach ($columns as $col): ?>
                                <td><?php echo esc_html($r->$col); ?></td>
                            <?php endforeach; ?>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr><td colspan="<?php echo count($columns); ?>">No records found</td></tr>
                <?php endif; ?>
            </tbody>
        </table>

        <!-- PAGINATION -->
        <?php
            $total_pages = ceil($total / $per_page);

            if ($total_pages > 1):
        ?>
        <div class="disp-pagination">
            <?php
                // PREVIOUS BUTTON
                if ($page > 1) {
                    echo '<a href="' . esc_url(add_query_arg([
                        'paged' => $page - 1,
                        'entries' => $per_page,
                        'search' => $search
                    ])) . '">Previous</a>';
                }

                // PAGE NUMBERS
                $start = max(1, $page - 2);
                $end   = min($total_pages, $page + 2);

                for ($i = $start; $i <= $end; $i++) {
                    echo '<a class="' . ($i==$page ? 'current' : '') . '" href="' . esc_url(add_query_arg([
                        'paged' => $i,
                        'entries' => $per_page,
                        'search' => $search
                    ])) . '">' . $i . '</a>';
                }

                // NEXT BUTTON
                if ($page < $total_pages) {
                    echo '<a href="' . esc_url(add_query_arg([
                        'paged' => $page + 1,
                        'entries' => $per_page,
                        'search' => $search
                    ])) . '">Next</a>';
                }
            ?>
        </div>
        <?php endif; ?>

    </div>

    <?php
    return ob_get_clean();
}

// ======================================================
// SHORTCODES
// ======================================================

// Homeopathic
function disp_homeopathic_sc() {
    return disp_generic_table_viewer(
        "homeopathic_disp",
        ["sl_no", "dhmo_office", "ghd_name", "district", "sub_division", "block_name", "gram_panchayat"]
    );
}
add_shortcode('homeopathic_disp_table', 'disp_homeopathic_sc');

// Ayurvedic
function disp_ayurvedic_sc() {
    return disp_generic_table_viewer(
        "ayurvedic_disp",
        ["sl_no", "damo_office", "gad", "district", "sub_division", "block_name", "gram_panchayat"]
    );
}
add_shortcode('ayurvedic_disp_table', 'disp_ayurvedic_sc');

// Unani
function disp_unani_sc() {
    return disp_generic_table_viewer(
        "unani_disp",
        ["sl_no", "circle_office", "uhd_name", "district", "sub_division", "block_name", "gram_panchayat"]
    );
}
add_shortcode('unani_disp_table', 'disp_unani_sc');

?>
