<?php
/*
Plugin Name: Browser Refresh Attack Protection
Description: Prevent Browser Refresh Attacks on login forms by implementing nonce validation, session control, and anti-caching.
Version: 1.0
Author: Cent Avenue
Author URI: https://centavenue.com/
*/

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Hook to start the plugin functionality on WordPress initialization
add_action('init', 'bra_prevent_cache');

// Prevent caching on the login page
function bra_prevent_cache() {
    if (is_page('wp-login.php')) {
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
    }
}

// Add nonce field to the login form
add_action('login_form', 'bra_add_nonce_to_login_form');

function bra_add_nonce_to_login_form() {
    wp_nonce_field('login_form_nonce_action', 'login_form_nonce_field');
}

// Validate nonce and handle form submission
add_filter('authenticate', 'bra_validate_nonce_on_login', 30, 3);

function bra_validate_nonce_on_login($user, $username, $password) {
    // Check if nonce field is set
    if (isset($_POST['login_form_nonce_field']) && wp_verify_nonce($_POST['login_form_nonce_field'], 'login_form_nonce_action')) {
        return $user; // Nonce verified, proceed with normal authentication
    } else {
        // Nonce verification failed
        return new WP_Error('nonce_failed', __('Security check failed. Please refresh and try again.'));
    }
}

// Regenerate the login session to prevent reuse
add_action('wp_login', 'bra_regenerate_login_session', 10, 2);

function bra_regenerate_login_session($user_login, $user) {
    // Regenerate the session ID to invalidate the old one
    session_regenerate_id(true);
}

// Force logout on session expiration to avoid resubmission attacks
add_action('init', 'bra_force_logout_on_expire');

function bra_force_logout_on_expire() {
    if (is_user_logged_in()) {
        // Check session expiration time
        $session_expiration = get_user_meta(get_current_user_id(), '_session_expire', true);
        if (time() > $session_expiration) {
            wp_logout(); // Force logout
            wp_redirect(wp_login_url()); // Redirect to login page
            exit;
        }
    }
}

// Set session expiration when a user logs in
add_action('wp_login', 'bra_set_session_expiration', 10, 2);

function bra_set_session_expiration($user_login, $user) {
    $session_lifetime = 3600; // Set session lifetime to 1 hour
    update_user_meta($user->ID, '_session_expire', time() + $session_lifetime);
}

?>
