<?php
/*
Plugin Name: Advanced Login Control
Description: Limits login attempts, tracks login attempts, handles lockout durations, and logs out users after a period of inactivity.
Version: 1.0
Author: Cent Avenue
Author URI: https://centavenue.com/
*/

// Hook into the login process
add_action('wp_login_failed', 'alc_login_failed');
add_action('wp_login', 'alc_login_success');

// Hook into the authentication process to check for lockouts
add_filter('authenticate', 'alc_authenticate', 30, 3);

// Add admin menu for plugin settings
add_action('admin_menu', 'alc_admin_menu');

// Register settings
add_action('admin_init', 'alc_register_settings');

// Add JavaScript to handle idle logout
add_action('wp_footer', 'alc_idle_logout_script');

// Function to handle failed login attempts
function alc_login_failed($username) {
    $attempts = get_transient('alc_login_attempts_' . $username);
    if ($attempts === false) {
        $attempts = 1;
    } else {
        $attempts++;
    }
    set_transient('alc_login_attempts_' . $username, $attempts, 12 * HOUR_IN_SECONDS);

    // Check if the user should be locked out
    $max_attempts = get_option('alc_max_attempts', 5);
    if ($attempts >= $max_attempts) {
        $lockout_duration = get_option('alc_lockout_duration', 60);
        set_transient('alc_lockout_' . $username, true, $lockout_duration * MINUTE_IN_SECONDS);
    }
}

// Function to handle successful logins
function alc_login_success($user_login) {
    delete_transient('alc_login_attempts_' . $user_login);
}

// Function to check lockout status during authentication
function alc_authenticate($user, $username, $password) {
    if (get_transient('alc_lockout_' . $username)) {
        return new WP_Error('too_many_attempts', __('Too many login attempts. Please try again later.'));
    }
    return $user;
}

// Admin menu for plugin settings
function alc_admin_menu() {
    add_options_page('Advanced Login Control Settings', 'Advanced Login Control', 'manage_options', 'advanced-login-control', 'alc_settings_page');
}

// Settings page content
function alc_settings_page() {
    ?>
    <div class="wrap">
        <h1>Advanced Login Control Settings</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('alc_settings_group');
            do_settings_sections('advanced-login-control');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// Register plugin settings
function alc_register_settings() {
    register_setting('alc_settings_group', 'alc_max_attempts');
    register_setting('alc_settings_group', 'alc_lockout_duration');
    register_setting('alc_settings_group', 'alc_idle_timeout');

    add_settings_section('alc_settings_section', 'Login Control Settings', null, 'advanced-login-control');

    add_settings_field('alc_max_attempts', 'Maximum Login Attempts', 'alc_max_attempts_field', 'advanced-login-control', 'alc_settings_section');
    add_settings_field('alc_lockout_duration', 'Lockout Duration (minutes)', 'alc_lockout_duration_field', 'advanced-login-control', 'alc_settings_section');
    add_settings_field('alc_idle_timeout', 'Idle Timeout (minutes)', 'alc_idle_timeout_field', 'advanced-login-control', 'alc_settings_section');
}

// Display max attempts field
function alc_max_attempts_field() {
    $value = get_option('alc_max_attempts', 5);
    echo '<input type="number" name="alc_max_attempts" value="' . esc_attr($value) . '" />';
}

// Display lockout duration field
function alc_lockout_duration_field() {
    $value = get_option('alc_lockout_duration', 60);
    echo '<input type="number" name="alc_lockout_duration" value="' . esc_attr($value) . '" />';
}

// Display idle timeout field
function alc_idle_timeout_field() {
    $value = get_option('alc_idle_timeout', 15);
    echo '<input type="number" name="alc_idle_timeout" value="' . esc_attr($value) . '" />';
}

// Add JavaScript for idle logout
function alc_idle_logout_script() {
    $idle_timeout = get_option('alc_idle_timeout', 15);
    ?>
    <script>
        (function($) {
            var idleTime = 0;
            var idleLimit = <?php echo intval($idle_timeout) * 60; ?>; // Convert minutes to seconds

            function resetTimer() {
                idleTime = 0;
            }

            function logoutUser() {
                window.location.href = '<?php echo wp_logout_url(); ?>';
            }

            $(document).ready(function() {
                setInterval(function() {
                    idleTime++;
                    if (idleTime > idleLimit) {
                        logoutUser();
                    }
                }, 1000);

                $(this).mousemove(resetTimer);
                $(this).keypress(resetTimer);
            });
        })(jQuery);
    </script>
    <?php
}
